import { type NextRequest, NextResponse } from "next/server"
import { ObjectId } from "mongodb"
import { getBranchesCollection, getUsersCollection } from "@/lib/database/collections"
import type { UpdateBranchInput } from "@/lib/models/Branch"

export async function GET(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const branchesCollection = await getBranchesCollection()
    const branch = await branchesCollection.findOne({ _id: new ObjectId(params.id) })

    if (!branch) {
      return NextResponse.json({ success: false, error: "Branch not found" }, { status: 404 })
    }

    // Populate owner and manager names
    const usersCollection = await getUsersCollection()
    const owner = branch.ownerId ? await usersCollection.findOne({ _id: branch.ownerId }) : null
    const manager = branch.managerId ? await usersCollection.findOne({ _id: branch.managerId }) : null

    const branchWithUsers = {
      ...branch,
      ownerName: owner?.name || null,
      managerName: manager?.name || null,
    }

    return NextResponse.json({ success: true, data: branchWithUsers })
  } catch (error) {
    console.error("Error fetching branch:", error)
    return NextResponse.json({ success: false, error: "Failed to fetch branch" }, { status: 500 })
  }
}

export async function PUT(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const body: UpdateBranchInput = await request.json()

    const updateData = {
      name: body.name,
      address: body.address,
      country: body.country,
      location: body.location,
      ownerId: body.ownerId ? new ObjectId(body.ownerId) : undefined,
      managerId: body.managerId ? new ObjectId(body.managerId) : undefined,
      isActive: body.isActive,
      updatedAt: new Date(),
    }

    const branchesCollection = await getBranchesCollection()
    const result = await branchesCollection.updateOne({ _id: new ObjectId(params.id) }, { $set: updateData })

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "Branch not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, data: updateData })
  } catch (error) {
    console.error("Error updating branch:", error)
    return NextResponse.json({ success: false, error: "Failed to update branch" }, { status: 500 })
  }
}

export async function DELETE(request: NextRequest, { params }: { params: { id: string } }) {
  try {
    const branchesCollection = await getBranchesCollection()

    // Soft delete by setting isActive to false instead of hard delete
    const result = await branchesCollection.updateOne(
      { _id: new ObjectId(params.id) },
      {
        $set: {
          isActive: false,
          updatedAt: new Date(),
        },
      },
    )

    if (result.matchedCount === 0) {
      return NextResponse.json({ success: false, error: "Branch not found" }, { status: 404 })
    }

    return NextResponse.json({ success: true, message: "Branch deactivated successfully" })
  } catch (error) {
    console.error("Error deleting branch:", error)
    return NextResponse.json({ success: false, error: "Failed to delete branch" }, { status: 500 })
  }
}
